#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

double calcJD(int year, int month, int day) {
	
	if (month <= 2) {
		year -= 1;
		month += 12;
	}

	double a = floor(year/100);
	double b = 2 - a + floor(a/4);

	double jd = floor(365.25 * (year + 4716)) + floor(30.6001 * (month +1)) + day + b - 1524.5;
	return jd;
}

double calcJDFromJulianCent(double t) {
	return (t * 36525 + 2451545);
}

double calcTimeJulianCent(double jd) {
	return ((jd - 2451545.0) / 36525.0);
}

double degToRad(double angleDeg) {
	return (M_PI * angleDeg / 180.0);
}

double radToDeg(double angleRad) {
	return (180.0 * angleRad / M_PI);
}

double calcMeanObliquityOfEcliptic(double t) {
	
	double seconds = 21.448 - t * (46.8150 + t * (0.00059 - t * (0.001813)));
	return 23.0 + (26.0 + (seconds / 60.0)) / 60.0; 
}

double calcObliquityCorrection(double t) {

	double e0 = calcMeanObliquityOfEcliptic(t);
	
	double omega = 125.04 - 1934.136 * t;
	return e0 + 0.00256 * cos(degToRad(omega));
}

double calcGeomMeanLongSun(double t) {

	double l0 = 280.46646 + t * (36000.76983 + 0.0003032 * t);

	while(l0 > 360.0) {
		l0 -= 360.0;
	}

	while(l0 < 0.0) {
		l0 += 360.0;
	}

	return l0;	// in degrees
}

double calcEccentricityEarthOrbit(double t) {
	return 0.016708634 - t * (0.000042037 + 0.0000001267 * t);
}

double calcGeomMeanAnomalySun(double t) {
	return 357.52911 + t * (35999.05029 - 0.0001537 * t);
}

double calcSunEqOfCenter(double t) {

	double m = calcGeomMeanAnomalySun(t);

	double mrad = degToRad(m);
	double sinm = sin(mrad);
	double sin2m = sin(2 * mrad);
	double sin3m = sin(3 * mrad);

	double c = sinm * (1.914602 - t * (0.004817 + 0.000014 * t)) + sin2m * (0.019993 - 0.000101 * t) + sin3m * 0.000289;
	return c;
}

double calcSunTrueLong(double t) {
	double l0 = calcGeomMeanLongSun(t);
	double c = calcSunEqOfCenter(t);

	return l0 + c;
}

double calcEquationOfTime(double t) {

	double epsilon = calcObliquityCorrection(t);
	double l0 = calcGeomMeanLongSun(t);
	double e = calcEccentricityEarthOrbit(t);
	double m = calcGeomMeanAnomalySun(t);

	double y = tan(degToRad(epsilon) / 2.0);
	y *= y;

	double sin2l0 = sin(2.0 * degToRad(l0));
	double sinm   = sin(degToRad(m));
	double cos2l0 = cos(2.0 * degToRad(l0));
	double sin4l0 = sin(4.0 * degToRad(l0));
	double sin2m  = sin(2.0 * degToRad(m));
	
	double etime = (y * sin2l0) - 2.0 * (e * sinm) + 4.0 * (e * y) * (sinm * cos2l0) - 0.5 * (y * y * sin4l0) - 1.25 * (e * e * sin2m);

	return radToDeg(etime) * 4.0;
}

double calcSunApparentLong(double t) {

	double o = calcSunTrueLong(t);

	double omega = 125.04 - 1934.136 * t;
	double lambda = o - 0.00569 - 0.00478 * sin(degToRad(omega));
	return lambda;
}

double calcSunRtAscension(double t) {

	double e = calcObliquityCorrection(t);
	double lambda = calcSunApparentLong(t);

	double tananum = cos(degToRad(e)) * sin(degToRad(lambda));
	double tanadenom = cos(degToRad(lambda));
	double alpha = radToDeg(atan2(tananum, tanadenom));
	return alpha;
}

double calcSunDeclination(double t) {

	double e = calcObliquityCorrection(t);
	double lambda = calcSunApparentLong(t);

	double sint = sin(degToRad(e)) * sin(degToRad(lambda));
	double theta = radToDeg(asin(sint));
	return theta;
}

double calcHourAngleSunrise(double latitude, double solarDec) {

	double latRad = degToRad(latitude);
	double sdRad = degToRad(solarDec);

	double haarg = (cos(degToRad(90.833)) / (cos(latRad) * cos(sdRad)) - tan(latRad) * tan(sdRad));

	double ha = acos(cos(degToRad(90.833)) / (cos(latRad) * cos(sdRad)) - tan(latRad) * tan(sdRad));

	return ha;	// In radians
}

double calcHourAngleSunset(double latitude, double solarDec) {

	double latRad = degToRad(latitude);
	double sdRad = degToRad(solarDec);

	double haarg = (cos(degToRad(90.883)) / (cos(latRad) * cos(sdRad)) - tan(latRad) * tan(sdRad));

	double ha = acos(cos(degToRad(90.833)) / (cos(latRad) * cos(sdRad)) - tan(latRad) * tan(sdRad));

	return 0 - ha;	// In radians
}

double calcSolNoonUTC(double t, double longitude) {

	double tnoon = calcTimeJulianCent(calcJDFromJulianCent(t) + longitude / 360.0);
	double eqTime = calcEquationOfTime(tnoon);
	double solNoonUTC = 720 + (longitude * 4) - eqTime;

	double newt = calcTimeJulianCent(calcJDFromJulianCent(t) - 0.5 + solNoonUTC / 1440.0);

	eqTime = calcEquationOfTime(newt);
	
	solNoonUTC = 720 + (longitude * 4) - eqTime;

	return solNoonUTC;
}

double calcSunriseUTC (double jd, double latitude, double longitude) {

	double t = calcTimeJulianCent(jd);

	// *** Find the time of solar noon at the location, and use
    //     that declination. This is better than start of the 
    //     Julian day

	double noonmin = calcSolNoonUTC(t, longitude);
	double tnoon = calcTimeJulianCent(jd + noonmin / 1440.0);

	// First pass to approximate sunrise (using solar noon)

	double eqTime = calcEquationOfTime(tnoon);
	double solarDec = calcSunDeclination(tnoon);
	double hourAngle = calcHourAngleSunrise(latitude, solarDec);

	double delta = longitude - radToDeg(hourAngle);
	double timeDiff = 4 * delta;
	double timeUTC = 720 + timeDiff - eqTime; // in minutes

	// Second pass includes fractional jday in gamma calc

	double newt = calcTimeJulianCent(calcJDFromJulianCent(t) + timeUTC / 1440);
	eqTime = calcEquationOfTime(newt);
	solarDec = calcSunDeclination(newt);
	hourAngle = calcHourAngleSunrise(latitude, solarDec);
	delta = longitude - radToDeg(hourAngle);
	timeDiff = 4 * delta;
	timeUTC = 720 + timeDiff - eqTime; // in minutes

	return timeUTC;
}

double calcSunsetUTC (double jd, double latitude, double longitude) {

	double t = calcTimeJulianCent(jd);

	// *** Find the time of solar noon at the location, and use
    //     that declination. This is better than start of the 
    //     Julian day

	double noonmin = calcSolNoonUTC(t, longitude);
	double tnoon = calcTimeJulianCent(jd + noonmin / 1440.0);

	// First calculates sunrise and approx. length of day

	double eqTime = calcEquationOfTime(tnoon);
	double solarDec = calcSunDeclination(tnoon);
	double hourAngle = calcHourAngleSunset(latitude, solarDec);

	double delta = longitude - radToDeg(hourAngle);
	double timeDiff = 4 * delta;
	double timeUTC = 720 + timeDiff - eqTime;
	
	// Second pass includes fractional jday in gamma calc

	double newt = calcTimeJulianCent(calcJDFromJulianCent(t) + timeUTC / 1440.0);
	eqTime = calcEquationOfTime(newt);
	solarDec = calcSunDeclination(newt);
	hourAngle = calcHourAngleSunset(latitude, solarDec);

	delta = longitude - radToDeg(hourAngle);
	timeDiff  = 4 * delta;
	timeUTC = 720 + timeDiff - eqTime;

	return timeUTC;
}

void usage(unsigned int val) {

	printf("Sunrise/Sunset time calculation. \n\n");
	printf("Usage: solar YYYY mm dd lat lon [tz offset]\n\n");
	printf("Example: solar 2015 4 8 41.85 87.65 -6\n\n");
}

int main(int argc, char *argv[]) {

	if (argc > 1) {

		int i, year, month, day = 0;
		double lat, lon, offset = 0;

		unsigned int val = 0;

		for (i = 0; i < argc; i++) {

			if (!strcmp(argv[i], "-l")) {
				printf("Locations:\n");
				printf("'US CITIES',0,0,0\n");
				printf("'Albuquerque, NM', 35.0833,106.65,7\n");
				printf("'Anchorage, AK', 61.217, 149.90,9\n");
				printf("'Atlanta, GA', 33.733, 84.383, 5\n");
				printf("'Austin, TX', 30.283, 97.733, 6\n");
				printf("'Birmingham, AL', 33.521, 86.8025, 6\n");
				printf("'Bismarck, ND', 46.817, 100.783, 6\n");
				printf("'Boston, MA', 42.35, 71.05, 5\n");
				printf("'Boulder, CO', 40.125, 105.237, 7\n");
				printf("'Chicago, IL', 41.85,87.65,6\n");
				printf("'Dallas, TX', 32.46, 96.47,6\n");
				printf("'Denver, CO', 39.733, 104.983, 7\n");
				printf("'Detroit, MI', 42.333, 83.05, 5\n");
				printf("'Honolulu, HI', 21.30, 157.85, 10\n");
				printf("'Houston, TX', 29.75, 95.35, 6\n");
				printf("'Indianapolis, IN', 39.767, 86.15, 5\n");
				printf("'Jackson, MS', 32.283, 90.183, 6\n");
				printf("'Kansas City, MO', 39.083, 94.567,6\n");
				printf("'Los Angeles, CA',34.05,118.233,8\n");
				printf("'Menomonee Falls, WI',43.11,88.10,6\n");
				printf("'Miami, FL', 25.767, 80.183,5\n");
				printf("'Minneapolis, MN', 44.967, 93.25, 6\n");
				printf("'New Orleans, LA', 29.95, 90.067, 6\n");
				printf("'New York City, NY', 40.7167, 74.0167, 5\n");
				printf("'Oklahoma City, OK', 35.483, 97.533,6\n");
				printf("'Philadelphia, PA', 39.95, 75.15, 5\n");
				printf("'Phoenix, AZ',33.433,112.067,7\n");
				printf("'Pittsburgh, PA',40.433,79.9833,5\n");
				printf("'Portland, ME', 43.666, 70.283, 5\n");
				printf("'Portland, OR', 45.517, 122.65, 8\n");
				printf("'Raleigh, NC', 35.783, 78.65, 5\n");
				printf("'Richmond, VA', 37.5667, 77.450, 5\n");
				printf("'Saint Louis, MO', 38.6167,90.1833,6\n");
				printf("'San Antonio, TX', 29.53, 98.47, 6\n");
				printf("'San Diego, CA', 32.7667, 117.2167, 8\n");
				printf("'San Francisco, CA',37.7667,122.4167,8\n");
				printf("'Seattle, WA',47.60,122.3167,8\n");
				printf("'Washington DC', 38.8833, 77.0333,5\n");
				printf("'',0,0,0\n");
				printf("'WORLD CITIES',0,0,0\n");
				printf("'Beijing, China',39.9167, -116.4167,-8\n");
				printf("'Berlin, Germany',52.33, -13.30, -1\n");
				printf("'Bombay, India', 18.9333, -72.8333, -5.5\n");
				printf("'Buenos Aires, Argentina', -34.60,58.45,3\n");
				printf("'Cairo, Egypt', 30.10,-31.3667,-2\n");
				printf("'Cape Town, South Africa',-33.9167,-18.3667,-2\n");
				printf("'Caracas, Venezuela', 10.50,66.9333,4\n");
				printf("'Helsinki, Finland', 60.1667, -24.9667,-2\n");
				printf("'Hong Kong, China', 22.25,-114.1667, -8\n");
				printf("'Jerusalem, Israel', 31.7833, -35.2333, -2\n");
				printf("'London, England', 51.50, 0.1667,0\n");
				printf("'Mexico City, Mexico', 19.4,99.15,6\n");
				printf("'Moscow, Russia', 55.75, -37.5833, -3\n");
				printf("'New Delhi, India',28.6, -77.2, -5.5\n");
				printf("'Ottawa, Canada', 45.41667,75.7,5\n");
				printf("'Paris, France', 48.8667, -2.667, -1\n");
				printf("'Rio de Janeiro, Brazil',-22.90,43.2333,3\n");
				printf("'Riyadh, Saudi Arabia', 24.633, -46.71667, -3\n");
				printf("'Rome, Italy',41.90, -12.4833,-1\n");
				printf("'Sydney, Australia',-33.8667,-151.2167,-10\n");
				printf("'Tokyo, Japan', 35.70, -139.7667, -9\n"); 
				printf("'Zurich, Switzerland', 47.3833, -8.5333,-1\n");
				printf("\n");
			}

			if (atoi(argv[i]) > 1900 && atoi(argv[i]) < 3000) {
				year = atoi(argv[i]);
				val |= (1 << 0);

				if (argc >= (i+2)) {
					if (atoi(argv[i+1]) > 0 && atoi(argv[i+1]) <= 12) {
						month = atoi(argv[i+1]);
						val |= (1 << 1);
					}
				}

				if (argc >= (i+3)) {
					if (atoi(argv[i+2]) > 0 && atoi(argv[i+2]) <= 31) {
						day = atoi(argv[i+2]);
						val |= (1 << 2);
					}
				}

				if (argc >= (i+4)) {
					if (atof(argv[i+3]) >= -90.0 && atof(argv[i+3]) <= 90.0) {
						lat = atof(argv[i+3]);
						val |= (1 << 3);
					}
				}

				if (argc >= (i+5)) {
					if (atof(argv[i+4]) >= -180.0 && atof(argv[i+4]) <= 180.00) {
						lon = atof(argv[i+4]);
						val |= (1 << 4);
					}
				}

				if (argc >= (i+6)) {
					if (atof(argv[i+5]) >= -24.0 && atof(argv[i+5]) <= 24.0) {
						offset = atof(argv[i+5]);
						val |= (1 << 5);
					}
				}

			}

		}

		if (val > 30) {

		double jd = calcJD(year, month, day);
		
		double sr = calcSunriseUTC(jd, lat, lon);
		double ss = calcSunsetUTC(jd, lat, lon);

		int srhr = floor(round(sr) / 60);
		srhr += offset;
		srhr %= 24;
		int srmin = round(sr);
		srmin %= 60;

		int sshr = floor(round(ss) / 60);
		sshr += offset;
		sshr %= 24;
		int ssmin = round(ss);
		ssmin %= 60;

		printf("Sunrise/Sunset: \n");
		printf ("%i/%i/%i\t", month, day, year);
		printf ("%02i:%02i\t", srhr, srmin);
		printf ("%02i:%02i\n", sshr, ssmin);

		} else if (val) {
			printf("Insufficient arguments. (index %i)\n\n", val);
			usage(0);
		}

	} else {

		usage(0);
	}

	return 0;
}
